'use client';

import { useEffect, useState } from 'react';
import StatCard from '@/components/dashboard/stat-card';
import { SalesChart, FuelDistributionChart } from '@/components/charts/sales-charts';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from '@/components/ui/select';
import {
  Fuel,
  TrendingUp,
  DollarSign,
  BarChart3,
  AlertCircle,
} from 'lucide-react';

interface DashboardData {
  overview: {
    totalTransactions: number;
    totalLiters: number;
    totalRevenue: number;
    avgLitersPerTx: number;
    avgRevenuePerTx: number;
  };
  salesByFuel: Array<{
    _id: string;
    totalLiters: number;
    totalRevenue: number;
    transactionCount: number;
    avgPrice: number;
  }>;
  dailyTrend: Array<{
    _id: string;
    totalRevenue: number;
    totalLiters: number;
    transactionCount: number;
  }>;
  paymentMethods: Array<{
    _id: string;
    count: number;
    totalAmount: number;
  }>;
}

export default function DashboardPage() {
  const [data, setData] = useState<DashboardData | null>(null);
  const [loading, setLoading] = useState(true);
  const [period, setPeriod] = useState('30');

  useEffect(() => {
    fetchDashboardData();
  }, [period]);

  const fetchDashboardData = async () => {
    try {
      setLoading(true);
      const response = await fetch(`/api/dashboard/overview?days=${period}`);
      const result = await response.json();
      if (result.success) {
        setData(result.data);
      }
    } catch (error) {
      console.error('Erreur chargement dashboard:', error);
    } finally {
      setLoading(false);
    }
  };

  if (loading) {
    return (
      <div className="flex h-full items-center justify-center">
        <div className="text-center">
          <div className="h-12 w-12 animate-spin rounded-full border-4 border-primary border-t-transparent mx-auto"></div>
          <p className="mt-4 text-muted-foreground">Chargement des données...</p>
        </div>
      </div>
    );
  }

  if (!data) {
    return (
      <div className="flex h-full items-center justify-center">
        <div className="text-center">
          <AlertCircle className="h-12 w-12 text-muted-foreground mx-auto" />
          <p className="mt-4 text-muted-foreground">Aucune donnée disponible</p>
        </div>
      </div>
    );
  }

  const chartData = data.dailyTrend.map((item) => ({
    date: item._id,
    totalRevenue: Math.round(item.totalRevenue),
    totalLiters: Math.round(item.totalLiters),
  }));

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold tracking-tight">Dashboard</h1>
          <p className="text-muted-foreground">
            Vue d'ensemble de l'activité de la station
          </p>
        </div>
        <Select value={period} onValueChange={setPeriod}>
          <SelectTrigger className="w-[180px]">
            <SelectValue placeholder="Période" />
          </SelectTrigger>
          <SelectContent>
            <SelectItem value="7">7 derniers jours</SelectItem>
            <SelectItem value="30">30 derniers jours</SelectItem>
            <SelectItem value="90">90 derniers jours</SelectItem>
          </SelectContent>
        </Select>
      </div>

      {/* KPIs */}
      <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
        <StatCard
          title="Chiffre d'Affaires"
          value={`${data.overview.totalRevenue.toLocaleString('fr-FR', {
            minimumFractionDigits: 2,
            maximumFractionDigits: 2,
          })} €`}
          subtitle={`${data.overview.totalTransactions.toLocaleString()} transactions`}
          icon={DollarSign}
        />
        <StatCard
          title="Volume Vendu"
          value={`${Math.round(data.overview.totalLiters).toLocaleString()} L`}
          subtitle="Tous carburants confondus"
          icon={Fuel}
        />
        <StatCard
          title="Panier Moyen"
          value={`${data.overview.avgRevenuePerTx.toLocaleString('fr-FR', {
            minimumFractionDigits: 2,
            maximumFractionDigits: 2,
          })} €`}
          subtitle={`${data.overview.avgLitersPerTx.toFixed(1)} L par transaction`}
          icon={BarChart3}
        />
        <StatCard
          title="Transactions"
          value={data.overview.totalTransactions.toLocaleString()}
          subtitle={`${Math.round(
            data.overview.totalTransactions / parseInt(period)
          )} transactions/jour`}
          icon={TrendingUp}
        />
      </div>

      {/* Charts */}
      <div className="grid gap-6 md:grid-cols-2">
        <Card>
          <CardHeader>
            <CardTitle>Évolution des Ventes</CardTitle>
          </CardHeader>
          <CardContent>
            <SalesChart data={chartData} />
          </CardContent>
        </Card>

        <Card>
          <CardHeader>
            <CardTitle>Répartition par Carburant</CardTitle>
          </CardHeader>
          <CardContent>
            <FuelDistributionChart data={data.salesByFuel} />
          </CardContent>
        </Card>
      </div>

      {/* Tables */}
      <Tabs defaultValue="fuel" className="space-y-4">
        <TabsList>
          <TabsTrigger value="fuel">Par Carburant</TabsTrigger>
          <TabsTrigger value="payment">Par Paiement</TabsTrigger>
        </TabsList>

        <TabsContent value="fuel" className="space-y-4">
          <Card>
            <CardHeader>
              <CardTitle>Détail par Type de Carburant</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                {data.salesByFuel.map((fuel) => (
                  <div
                    key={fuel._id}
                    className="flex items-center justify-between border-b pb-4 last:border-0"
                  >
                    <div>
                      <p className="font-medium">{fuel._id}</p>
                      <p className="text-sm text-muted-foreground">
                        {fuel.transactionCount} transactions
                      </p>
                    </div>
                    <div className="text-right">
                      <p className="font-medium">
                        {fuel.totalRevenue.toLocaleString('fr-FR', {
                          minimumFractionDigits: 2,
                        })}{' '}
                        €
                      </p>
                      <p className="text-sm text-muted-foreground">
                        {Math.round(fuel.totalLiters).toLocaleString()} L
                      </p>
                      <p className="text-xs text-muted-foreground">
                        Prix moy: {fuel.avgPrice.toFixed(4)} €/L
                      </p>
                    </div>
                  </div>
                ))}
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="payment" className="space-y-4">
          <Card>
            <CardHeader>
              <CardTitle>Répartition par Mode de Paiement</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                {data.paymentMethods.map((method) => (
                  <div
                    key={method._id}
                    className="flex items-center justify-between border-b pb-4 last:border-0"
                  >
                    <div>
                      <p className="font-medium">{method._id}</p>
                      <p className="text-sm text-muted-foreground">
                        {method.count} transactions
                      </p>
                    </div>
                    <div className="text-right">
                      <p className="font-medium">
                        {method.totalAmount.toLocaleString('fr-FR', {
                          minimumFractionDigits: 2,
                        })}{' '}
                        €
                      </p>
                      <p className="text-sm text-muted-foreground">
                        {(
                          (method.count / data.overview.totalTransactions) *
                          100
                        ).toFixed(1)}
                        % des transactions
                      </p>
                    </div>
                  </div>
                ))}
              </div>
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>
    </div>
  );
}
