import { NextResponse } from 'next/server';
import connectDB from '@/lib/mongodb';
import { Tank, TankReading } from '@/models';

interface TankReadingDoc {
  levelLiters: number;
  readingDatetime: Date;
  tankId: unknown;
  temperature?: number;
  density?: number;
}

/**
 * PUT /api/tanks/update-level
 * Met à jour manuellement le niveau d'une citerne
 */
export async function PUT(request: Request) {
  try {
    await connectDB();

    const body = await request.json();
    const { tankId, newLevel, readingType = 'MANUAL' } = body;

    console.log('🔧 UPDATE TANK LEVEL - Requête reçue:', { tankId, newLevel, readingType });

    // Validation
    if (!tankId || newLevel === undefined || newLevel < 0) {
      return NextResponse.json(
        {
          success: false,
          error: 'Données invalides. Le niveau doit être >= 0.',
        },
        { status: 400 }
      );
    }

    // 1. Récupérer la citerne
    const tank = await Tank.findById(tankId);
    if (!tank) {
      return NextResponse.json(
        {
          success: false,
          error: 'Citerne introuvable',
        },
        { status: 404 }
      );
    }

    // 2. Vérifier que le niveau ne dépasse pas la capacité
    if (newLevel > tank.maxLevelLiters) {
      return NextResponse.json(
        {
          success: false,
          error: `Le niveau (${newLevel}L) dépasse la capacité maximale (${tank.maxLevelLiters}L)`,
        },
        { status: 400 }
      );
    }

    // 3. Récupérer le dernier relevé pour comparaison
    const lastReading = await TankReading.findOne({ tankId: tank._id })
      .sort({ readingDatetime: -1 })
      .lean() as TankReadingDoc | null;

    // 4. Créer un nouveau relevé avec le niveau manuel
    const newReading = await TankReading.create({
      tankId: tank._id,
      readingDatetime: new Date(),
      levelLiters: newLevel,
      temperature: lastReading?.temperature || 15,
      density: lastReading?.density || 0.85,
      readingType: readingType,
    });

    return NextResponse.json({
      success: true,
      message: 'Niveau de citerne mis à jour avec succès',
      data: {
        tankCode: tank.tankCode,
        previousLevel: lastReading?.levelLiters || 0,
        newLevel: newLevel,
        difference: newLevel - (lastReading?.levelLiters || 0),
        reading: newReading,
      },
    });
  } catch (error) {
    console.error('Erreur API Tank Update Level:', error);
    return NextResponse.json(
      {
        success: false,
        error: error instanceof Error ? error.message : 'Erreur lors de la mise à jour du niveau',
      },
      { status: 500 }
    );
  }
}
