import { NextResponse } from 'next/server';
import connectDB from '@/lib/mongodb';
import { Tank, TankReading } from '@/models';

interface TankReadingDoc {
  levelLiters: number;
  readingDatetime: Date;
  tankId: unknown;
}

/**
 * GET /api/tanks
 * Retourne toutes les citernes avec leurs niveaux actuels
 */
export async function GET() {
  try {
    await connectDB();

    const tanks = await Tank.find({ active: true }).lean();

    // Pour chaque citerne, récupérer le dernier relevé
    const tanksWithLevels = await Promise.all(
      tanks.map(async (tank) => {
        const lastReading = await TankReading.findOne({ tankId: tank._id })
          .sort({ readingDatetime: -1 })
          .lean() as TankReadingDoc | null;

        return {
          _id: tank._id,
          tankCode: tank.tankCode,
          fuelType: tank.fuelType,
          capacityLiters: tank.capacityLiters,
          minLevelLiters: tank.minLevelLiters,
          maxLevelLiters: tank.maxLevelLiters,
          currentLevel: lastReading?.levelLiters || 0,
          lastReadingDate: lastReading?.readingDatetime || null,
          fillPercentage: lastReading ? (lastReading.levelLiters / tank.capacityLiters) * 100 : 0,
        };
      })
    );

    return NextResponse.json({
      success: true,
      data: tanksWithLevels,
    });
  } catch (error) {
    console.error('Erreur API Tanks:', error);
    return NextResponse.json(
      {
        success: false,
        error: 'Erreur lors de la récupération des citernes',
      },
      { status: 500 }
    );
  }
}
