import { NextResponse } from 'next/server';
import connectDB from '@/lib/mongodb';
import { Supplier } from '@/models';

/**
 * GET /api/suppliers
 * Retourne tous les fournisseurs
 */
export async function GET() {
  try {
    await connectDB();

    const suppliers = await Supplier.find().sort({ supplierName: 1 }).lean();

    return NextResponse.json({
      success: true,
      data: suppliers,
    });
  } catch (error) {
    console.error('Erreur API Suppliers GET:', error);
    return NextResponse.json(
      {
        success: false,
        error: 'Erreur lors de la récupération des fournisseurs',
      },
      { status: 500 }
    );
  }
}

/**
 * POST /api/suppliers
 * Crée un nouveau fournisseur
 */
export async function POST(request: Request) {
  try {
    await connectDB();

    const body = await request.json();
    const { 
      supplierCode, 
      supplierName, 
      category, 
      contactPerson, 
      phone, 
      email, 
      address, 
      city, 
      postalCode, 
      taxId, 
      paymentTerms 
    } = body;

    // Validation
    if (!supplierCode || !supplierName || !category) {
      return NextResponse.json(
        {
          success: false,
          error: 'Code, nom et catégorie sont obligatoires',
        },
        { status: 400 }
      );
    }

    // Vérifier si le code existe déjà
    const existing = await Supplier.findOne({ supplierCode });
    if (existing) {
      return NextResponse.json(
        {
          success: false,
          error: `Le code fournisseur "${supplierCode}" existe déjà`,
        },
        { status: 400 }
      );
    }

    // Créer le fournisseur
    const supplier = await Supplier.create({
      supplierCode,
      supplierName,
      category,
      contactPerson,
      phone,
      email,
      address,
      city,
      postalCode,
      taxId,
      paymentTerms,
      active: true,
    });

    return NextResponse.json({
      success: true,
      message: 'Fournisseur créé avec succès',
      data: supplier,
    });
  } catch (error) {
    console.error('Erreur API Suppliers POST:', error);
    return NextResponse.json(
      {
        success: false,
        error: error instanceof Error ? error.message : 'Erreur lors de la création du fournisseur',
      },
      { status: 500 }
    );
  }
}
