import { NextResponse } from 'next/server';
import connectDB from '@/lib/mongodb';
import { PumpTransaction, TankReading, Tank, Pump, Product } from '@/models';

/**
 * Détermine le shift (quart de travail) en fonction de l'heure
 */
function getShift(): 'MORNING' | 'AFTERNOON' | 'NIGHT' {
  const hour = new Date().getHours();
  if (hour >= 6 && hour < 14) return 'MORNING';
  if (hour >= 14 && hour < 22) return 'AFTERNOON';
  return 'NIGHT';
}

/**
 * POST /api/sales/create
 * Enregistre une nouvelle vente et décrémente le stock de la citerne
 */
export async function POST(request: Request) {
  try {
    await connectDB();

    const body = await request.json();
    const { pumpId, fuelType, counterStart, counterEnd, unitPrice, customerId, paymentMethod } = body;

    // Calcul automatique des litres par déduction
    const liters = counterEnd - counterStart;

    // Validation des données
    if (!pumpId || !fuelType || !counterStart || !counterEnd || counterEnd <= counterStart) {
      return NextResponse.json(
        {
          success: false,
          error: 'Données invalides. Le compteur final doit être supérieur au compteur initial.',
        },
        { status: 400 }
      );
    }

    // Calcul du montant total
    const totalAmount = liters * unitPrice;

    // 1. Récupérer la pompe pour obtenir le compteur cumulatif et mise à jour
    const pump = await Pump.findById(pumpId);
    if (!pump) {
      return NextResponse.json(
        {
          success: false,
          error: 'Pompe introuvable',
        },
        { status: 404 }
      );
    }

    // Vérifier que le compteur initial correspond au compteur cumulatif actuel
    if (Math.abs(counterStart - pump.cumulativeCounter) > 0.1) {
      return NextResponse.json(
        {
          success: false,
          error: `Le compteur initial (${counterStart}L) ne correspond pas au compteur actuel de la pompe (${pump.cumulativeCounter}L)`,
        },
        { status: 400 }
      );
    }

    // 2. Trouver le produit (carburant) correspondant au fuelType
    const product = await Product.findOne({ fuelType, active: true });
    if (!product) {
      return NextResponse.json(
        {
          success: false,
          error: `Aucun produit trouvé pour ${fuelType}`,
        },
        { status: 404 }
      );
    }

    // 3. Trouver la citerne correspondante au type de carburant
    // Note: Pour simplifier, on cherche la citerne par fuelType
    // Dans une version plus avancée, on utiliserait tank_mapping
    const tank = await Tank.findOne({ fuelType, active: true });
    if (!tank) {
      return NextResponse.json(
        {
          success: false,
          error: `Aucune citerne active trouvée pour ${fuelType}`,
        },
        { status: 404 }
      );
    }

    // 4. Récupérer le dernier relevé de la citerne
    const lastReading = await TankReading.findOne({ tankId: tank._id })
      .sort({ readingDatetime: -1 })
      .lean();

    if (!lastReading) {
      return NextResponse.json(
        {
          success: false,
          error: 'Aucun relevé de citerne trouvé',
        },
        { status: 404 }
      );
    }

    // Vérifier qu'il y a assez de stock
    // @ts-expect-error - Mongoose lean() type issue
    const newLevel = lastReading.levelLiters - liters;
    if (newLevel < tank.minLevelLiters) {
      return NextResponse.json(
        {
          success: false,
          // @ts-expect-error - Mongoose lean() type issue
          error: `Stock insuffisant. Niveau actuel: ${lastReading.levelLiters.toFixed(2)}L, minimum requis: ${tank.minLevelLiters}L`,
        },
        { status: 400 }
      );
    }

    // 5. Créer la transaction de vente
    const transaction = await PumpTransaction.create({
      txDatetime: new Date(),
      pumpId: pumpId,
      productId: product._id, // Utiliser le produit trouvé
      fuelType: fuelType,
      counterStart: counterStart,
      counterEnd: counterEnd,
      liters: liters,
      unitPrice: unitPrice,
      totalAmount: totalAmount,
      paymentMethod: paymentMethod,
      customerId: customerId === 'DIVERS' ? null : customerId,
      shift: getShift(),
    });

    // 5. Mettre à jour le compteur cumulatif de la pompe
    pump.cumulativeCounter = counterEnd;
    await pump.save();

    // 6. Créer un nouveau relevé de citerne avec le niveau décrémenté
    await TankReading.create({
      tankId: tank._id,
      readingDatetime: new Date(),
      levelLiters: newLevel,
      // @ts-expect-error - Mongoose lean() type issue
      temperature: lastReading.temperature,
      // @ts-expect-error - Mongoose lean() type issue
      density: lastReading.density,
      readingType: 'MANUAL',
    });

    return NextResponse.json({
      success: true,
      message: 'Vente enregistrée avec succès',
      data: {
        transaction,
        pumpUpdate: {
          pumpCode: pump.pumpCode,
          counterStart: counterStart,
          counterEnd: counterEnd,
          litersDispensed: liters,
        },
        tankUpdate: {
          tankCode: tank.tankCode,
          // @ts-expect-error - Mongoose lean() type issue
          previousLevel: lastReading.levelLiters,
          newLevel,
          remaining: newLevel,
        },
      },
    });
  } catch (error) {
    console.error('Erreur API Sales Create:', error);
    return NextResponse.json(
      {
        success: false,
        error: error instanceof Error ? error.message : 'Erreur lors de l\'enregistrement de la vente',
      },
      { status: 500 }
    );
  }
}
